SIF API - instruções para integração
==================================

Resumo
-----
Este serviço fornece dois endpoints principais usados pelos PDVs:

- `POST /api/register_logged_status` — regista estado de login (PDV envia `TaxRegistrationNumber`, `logged_status`, `licence`, `token`).
- `GET|POST /api/get_license_by_nif` — devolve o registo mais recente da tabela `tec_licences` para um NIF.

Segurança
--------
- Utilizamos HMAC (por defeito `sha256`) com um segredo partilhado (`sif_api_secret` em `app/config/sif_api.php`).
- Para `get_license_by_nif` é requerida proteção adicional (timestamp + nonce) para evitar replay. O token é calculado sobre o payload:

  payload = TaxRegistrationNumber + '|' + ts + '|' + nonce

  token = hash_hmac(algo, payload, secret)

- Para `register_logged_status` o token actual é calculado como:

  payload = TaxRegistrationNumber + '|' + logged_status + '|' + licence
  token = hash_hmac(algo, payload, secret)

Recomenda-se a actualização para incluir ts+nonce neste endpoint também.

Gerar token (exemplos)
----------------------

1) PHP (get_license_by_nif)

```php
<?php
$nif = '123456789';
$secret = 'COLOQUE_O_SEGREDO_AQUI';
$ts = time();
$nonce = bin2hex(random_bytes(16));
$payload = $nif . '|' . $ts . '|' . $nonce;
$token = hash_hmac('sha256', $payload, $secret);
echo "ts=$ts\nnonce=$nonce\ntoken=$token\n";
```

2) OpenSSL (linha de comando)

```bash
ts=$(date +%s)
nonce=$(openssl rand -hex 16)
payload="123456789|${ts}|${nonce}"
token=$(echo -n "$payload" | openssl dgst -sha256 -hmac "YOUR_SECRET" | sed 's/^.*= //')
echo $ts $nonce $token
```

3) Python

```python
import hmac,hashlib,os,time
nif='123456789'
secret=b'YOUR_SECRET'
ts=str(int(time.time()))
nonce=os.urandom(16).hex()
payload = f"{nif}|{ts}|{nonce}".encode()
token = hmac.new(secret, payload, hashlib.sha256).hexdigest()
print(ts, nonce, token)
```

Curl examples
-------------

GET (com querystring)

```bash
curl "https://mc.sifpos.ao/api/get_license_by_nif?TaxRegistrationNumber=123456789&ts=TIMESTAMP&nonce=NONCE&token=TOKEN"
```

POST JSON

```bash
curl -X POST https://mc.sifpos.ao/api/get_license_by_nif \
  -H "Content-Type: application/json" \
  -d '{"TaxRegistrationNumber":"123456789","ts":"TIMESTAMP","nonce":"NONCE","token":"TOKEN"}'
```

register_logged_status (form-post)

```bash
curl -X POST https://mc.sifpos.ao/api/register_logged_status \
  -d "TaxRegistrationNumber=123456789" \
  -d "logged_status=1" \
  -d "licence=ABCDEF" \
  -d "token=TOKEN"
```

Banco de dados
--------------

- Criar tabela para os registos do PDV (já incluída em `sql/create_tec_logged_status.sql`).
- Criar tabela para nonces (replay protection): `sql/create_sif_api_nonces.sql`.

Exemplo de execução SQL (MySQL):

```sql
SOURCE sql/create_tec_logged_status.sql;
SOURCE sql/create_sif_api_nonces.sql;
```

Configuração
-------------
- Edite `application/config/sif_api.php` com um segredo forte, por exemplo 32+ caracteres.
- As opções relevantes:
  - `sif_api_secret` — segredo HMAC
  - `sif_api_require_token` — true/false
  - `sif_api_require_nonce` — true/false (replay protection)
  - `sif_api_ts_window` — janela em segundos para aceitar timestamps

Notas finais
-----------
- Se o PDV e o servidor não partilharem exactamente o mesmo segredo, os tokens serão inválidos.
- Para debugging temporário, defina `sif_api_require_token = false` (não recomendado em produção).
