<?php defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * AGT / MinFin - Facturação Electrónica (SIGT)
 * CodeIgniter 3 Library
 *
 * Requisitos:
 * - PHP 7.4+
 * - OpenSSL habilitado
 *
 * Como usar no Controller:
 *   $this->load->library('AgtFeClient', $config);
 *   $res = $this->agtfeclient->registarFacturaFT($doc);
 */

class AgtFeClient
{
    /** @var CI_Controller */
    protected $ci;

    protected $env = 'homolog';

    // Endpoint base: tu podes passar o endpoint completo no config também
    protected $endpoint_registar = 'https://sifphml.minfin.gov.ao/sigt/fe/v1/registarFactura';

    // Token Bearer (se aplicável no teu cenário)
    protected $bearer_token = '';

    // Caminho do certificado/CA (opcional, mas recomendado)
    protected $ca_bundle_path = '';

    // Chaves privadas (PEM) para assinar
    protected $private_key_software_pem = '';
    protected $private_key_emissor_pem  = '';

    // Metadados do software
    protected $productId = 'SIF-POS';
    protected $productVersion = '1.2.2';
    protected $softwareValidationNumber = 'FE/132/AGT/2025';

    public function __construct($config = [])
    {
        $this->ci = &get_instance();

        foreach ($config as $k => $v) {
            if (property_exists($this, $k)) {
                $this->$k = $v;
            }
        }
    }

    /* =========================================================
     *  API PUBLICA
     * ========================================================= */

    /**
     * Registar Factura do tipo FT (Factura)
     * $doc esperado:
     * [
     *   'taxRegistrationNumber' => '5000...',
     *   'documentNo' => 'FT.../1',
     *   'documentDate' => '2025-12-12',
     *   'systemEntryDate' => '2025-12-12T18:38:09Z',
     *   'customerCountry' => 'AO',
     *   'customerTaxID' => '999999999',
     *   'companyName' => 'Cliente final',
     *   'lines' => [
     *      [
     *        'lineNumber'=>1,
     *        'productCode'=>'1001010',
     *        'productDescription'=>'Teste',
     *        'quantity'=>1,
     *        'unitOfMeasure'=>'UN',
     *        'unitPriceBase'=>11200,
     *        'unitPrice'=>11200,
     *        'settlementAmount'=>0,
     *        // IVA normal exemplo:
     *        'taxType'=>'IVA',
     *        'taxCountryRegion'=>'AO',
     *        'taxCode'=>'NOR',
     *        'taxPercentage'=>14
     *      ]
     *   ]
     * ]
     */
    public function registarFacturaFT(array $doc)
    {
        $nowIso = gmdate('Y-m-d\TH:i:s\Z');

        $taxRegistrationNumber = $doc['taxRegistrationNumber'];
        $submissionUUID = $this->uuidv4();
        $submissionTimeStamp = $nowIso;

        // 1) softwareInfoDetail
        $softwareInfoDetail = [
            'productId' => $this->productId,
            'productVersion' => $this->productVersion,
            'softwareValidationNumber' => $this->softwareValidationNumber,
        ];

        // 2) softwareInfo (assinado pelo produtor/software)
        $softwareInfo = [
            'softwareInfoDetail' => $softwareInfoDetail,
            'jwsSoftwareSignature' => null,
        ];

        $softwareInfo['jwsSoftwareSignature'] = $this->jwsSignRs256(
            $softwareInfo, // assina o object softwareInfo todo (conforme regra)
            $this->private_key_software_pem
        );

        // 3) montar documento (com lines + totals)
        $document = $this->buildFTDocument($doc, $taxRegistrationNumber);

        // 4) assinatura do documento (jwsDocumentSignature)
        // Campos a assinar incluem documentTotals, etc. :contentReference[oaicite:2]{index=2}
        $documentToSign = [
            'documentNo' => $document['documentNo'],
            'taxRegistrationNumber' => $taxRegistrationNumber,
            'documentType' => $document['documentType'],
            'documentDate' => $document['documentDate'],
            'customerTaxID' => $document['customerTaxID'],
            'customerCountry' => $document['customerCountry'],
            'companyName' => $document['companyName'],
            'documentTotals' => $document['documentTotals'],
        ];

        $document['jwsDocumentSignature'] = $this->jwsSignRs256(
            $documentToSign,
            $this->private_key_emissor_pem
        );

        // 5) payload base
        $payload = [
            'schemaVersion' => '1.0',
            'submissionUUID' => $submissionUUID,
            'taxRegistrationNumber' => $taxRegistrationNumber,
            'submissionTimeStamp' => $submissionTimeStamp,
            'softwareInfo' => $softwareInfo,
            'jwsSignature' => null,
            'numberOfEntries' => 1,
            'documents' => [ $document ],
        ];

        // 6) assinatura da chamada (jwsSignature) - assina submissionUUID + taxRegistrationNumber
        $callToSign = [
            'submissionUUID' => $payload['submissionUUID'],
            'taxRegistrationNumber' => $payload['taxRegistrationNumber'],
        ];

        $payload['jwsSignature'] = $this->jwsSignRs256(
            $callToSign,
            $this->private_key_emissor_pem
        );

        // 7) POST
        return $this->httpPostJson($this->endpoint_registar, $payload);
    }

    /* =========================================================
     *  BUILDERS
     * ========================================================= */

    protected function buildFTDocument(array $doc, $taxRegistrationNumber)
    {
        $documentNo = $doc['documentNo'];
        $documentDate = $doc['documentDate']; // YYYY-MM-DD
        $systemEntryDate = $doc['systemEntryDate'] ?? gmdate('Y-m-d\TH:i:s\Z');

        $customerCountry = $doc['customerCountry'] ?? 'AO';
        $customerTaxID   = $doc['customerTaxID'] ?? '999999999';
        $companyName     = $doc['companyName'] ?? 'Cliente final';

        $linesIn = $doc['lines'] ?? [];
        if (empty($linesIn)) {
            throw new Exception("FT precisa de lines (lista de artigos/serviços).");
        }

        $linesOut = [];
        $netTotal = 0.0;
        $taxPayable = 0.0;

        foreach ($linesIn as $ln) {
            $qty = (float)$ln['quantity'];
            $unitPriceBase = (float)$ln['unitPriceBase'];
            $unitPrice = (float)$ln['unitPrice'];
            $settlement = (float)($ln['settlementAmount'] ?? 0);

            // Base sem imposto (por linha)
            $lineBase = ($qty * $unitPriceBase);
            // Total da linha sem imposto (já com descontos se existirem)
            $lineNet = ($qty * $unitPrice) - $settlement;

            // IVA (exemplo: NOR 14%)
            $taxType = $ln['taxType'] ?? 'IVA';
            $taxCountryRegion = $ln['taxCountryRegion'] ?? 'AO';
            $taxCode = $ln['taxCode'] ?? 'NOR';
            $taxPercentage = (float)($ln['taxPercentage'] ?? 14);

            // Se for isento (ISE) ou NS, a AGT pede também taxExemptionCode (conforme regra do manual). :contentReference[oaicite:3]{index=3}
            $taxExemptionCode = $ln['taxExemptionCode'] ?? null;

            $taxContribution = 0.0;
            if ($taxType === 'IVA' && $taxCode === 'NOR') {
                $taxContribution = $this->roundUp2($lineNet * ($taxPercentage / 100));
            } elseif ($taxType === 'IVA' && $taxCode === 'ISE') {
                $taxPercentage = 0.0;
                $taxContribution = 0.0;
                // aqui, em produção, define taxExemptionCode conforme as tabelas do manual
                if (empty($taxExemptionCode)) {
                    // para não “morrer” em teste, mete um placeholder
                    $taxExemptionCode = 'M99';
                }
            } elseif ($taxType === 'NS') {
                $taxPercentage = 0.0;
                $taxContribution = 0.0;
                if (empty($taxExemptionCode)) {
                    $taxExemptionCode = 'M99';
                }
            } else {
                // default: sem imposto
                $taxPercentage = 0.0;
                $taxContribution = 0.0;
            }

            $taxObj = [
                'taxType' => $taxType,
                'taxCountryRegion' => $taxCountryRegion,
                'taxCode' => $taxCode,
                'taxPercentage' => $taxPercentage,
                'taxAmount' => 0,
                'taxContribution' => $taxContribution,
            ];
            if (!empty($taxExemptionCode)) {
                $taxObj['taxExemptionCode'] = $taxExemptionCode;
            }

            // Para FT: usa debitAmount (não creditAmount)
            $lineOut = [
                'lineNumber' => (int)$ln['lineNumber'],
                'productCode' => (string)$ln['productCode'],
                'productDescription' => (string)$ln['productDescription'],
                'quantity' => $qty,
                'unitOfMeasure' => (string)$ln['unitOfMeasure'],
                'unitPrice' => $unitPrice,
                'unitPriceBase' => $unitPriceBase,
                'debitAmount' => $this->round2($lineNet),
                'taxes' => [ $taxObj ],
                'settlementAmount' => $this->round2($settlement),
            ];

            $linesOut[] = $lineOut;

            $netTotal += $lineNet;
            $taxPayable += $taxContribution;
        }

        $netTotal = $this->round2($netTotal);
        $taxPayable = $this->round2($taxPayable);
        $grossTotal = $this->round2($netTotal + $taxPayable);

        // documentTotals é obrigatório :contentReference[oaicite:4]{index=4}
        $documentTotals = [
            'taxPayable' => $taxPayable,
            'netTotal' => $netTotal,
            'grossTotal' => $grossTotal,
        ];

        return [
            'documentNo' => $documentNo,
            'documentStatus' => 'N',
            'jwsDocumentSignature' => null,
            'documentDate' => $documentDate,
            'documentType' => 'FT',
            'systemEntryDate' => $systemEntryDate,
            'customerCountry' => $customerCountry,
            'customerTaxID' => $customerTaxID,
            'companyName' => $companyName,
            'lines' => $linesOut,
            'documentTotals' => $documentTotals,
        ];
    }

    /* =========================================================
     *  HTTP
     * ========================================================= */

    protected function httpPostJson($url, array $payload)
    {
        $json = json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);

        $ch = curl_init($url);
        $headers = [
            'Content-Type: application/json',
            'Accept: application/json',
        ];

        if (!empty($this->bearer_token)) {
            $headers[] = 'Authorization: Bearer ' . $this->bearer_token;
        }

        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $json,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CONNECTTIMEOUT => 20,
            CURLOPT_TIMEOUT => 60,
        ]);

        // SSL verify (recomendado em produção)
        if (!empty($this->ca_bundle_path) && file_exists($this->ca_bundle_path)) {
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_CAINFO, $this->ca_bundle_path);
        } else {
            // se estiveres a testar e não tens CA bundle, podes manter false temporariamente
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        }

        $body = curl_exec($ch);
        $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($body === false) {
            $err = curl_error($ch);
            curl_close($ch);
            return [
                'ok' => false,
                'http_code' => $http ?: 0,
                'body' => '',
                'error' => $err,
                'payload' => $payload,
            ];
        }

        curl_close($ch);

        return [
            'ok' => ($http >= 200 && $http < 300),
            'http_code' => $http,
            'body' => $body,
            'error' => '',
            'payload' => $payload,
        ];
    }

    /* =========================================================
     *  JWS (RS256)
     * ========================================================= */

    protected function jwsSignRs256(array $payload, $privateKeyPem)
    {
        if (empty($privateKeyPem)) {
            throw new Exception("Private key PEM não configurada para assinatura RS256.");
        }

        $header = ['alg' => 'RS256', 'typ' => 'JWT'];

        $segments = [];
        $segments[] = $this->base64UrlEncode(json_encode($header, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
        $segments[] = $this->base64UrlEncode(json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));

        $signingInput = implode('.', $segments);

        $pkey = openssl_pkey_get_private($privateKeyPem);
        if (!$pkey) {
            throw new Exception("Falha ao carregar private key PEM (openssl_pkey_get_private).");
        }

        $signature = '';
        $ok = openssl_sign($signingInput, $signature, $pkey, OPENSSL_ALGO_SHA256);
        openssl_free_key($pkey);

        if (!$ok) {
            throw new Exception("Falha ao assinar RS256 (openssl_sign).");
        }

        $segments[] = $this->base64UrlEncode($signature);

        return implode('.', $segments);
    }

    protected function base64UrlEncode($data)
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    /* =========================================================
     *  HELPERS
     * ========================================================= */

    protected function uuidv4()
    {
        $data = random_bytes(16);
        $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
        $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);
        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }

    protected function round2($v)
    {
        return round((float)$v, 2);
    }

    // arredondar por excesso para o cêntimo seguinte (como o manual descreve)
    protected function roundUp2($v)
    {
        $v = (float)$v;
        return ceil($v * 100) / 100;
    }
}
