<?php defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * API pública mínima para registar estado de login enviado por PDVs.
 * Endpoint: /api_sif/register_logged_status
 */
class Api_sif extends CI_Controller {

    public function __construct()
    {
        parent::__construct();
        $this->load->model('logged_model');
        $this->load->helper('security');
        $this->config->load('sif_api', true);
        $this->sif_secret = $this->config->item('sif_api_secret', 'sif_api');
        $this->sif_require_token = $this->config->item('sif_api_require_token', 'sif_api');
        $this->sif_algo = $this->config->item('sif_api_token_algo', 'sif_api') ?: 'sha256';
        $this->output->set_content_type('application/json');
    }

    /**
     * Regista estado de login vindo de clientes.
     * Aceita POST (form-url-encoded ou json) com pelo menos:
     * - TaxRegistrationNumber (NIF)
     * - logged_status (0/1)
     * Opcional: logged_ip, licence, token
     */
    public function register_logged_status()
    {
        // Aceitar apenas POST
        if ($this->input->method() !== 'post') {
            $this->output->set_status_header(405)->set_output(json_encode(['status' => 'error', 'message' => 'Método não permitido']));
            return;
        }

        // Suporta application/json
        $raw = trim(file_get_contents('php://input'));
        if (!empty($raw) && $this->input->server('CONTENT_TYPE') && strpos($this->input->server('CONTENT_TYPE'), 'application/json') !== false) {
            $posted = json_decode($raw, true);
            if (!is_array($posted)) $posted = array();
        } else {
            $posted = $this->input->post();
        }

        $nif = isset($posted['TaxRegistrationNumber']) ? $this->security->xss_clean(trim($posted['TaxRegistrationNumber'])) : null;
        $logged_status = isset($posted['logged_status']) ? (int)$posted['logged_status'] : null;
        $logged_ip = isset($posted['logged_ip']) ? $this->security->xss_clean(trim($posted['logged_ip'])) : $this->input->ip_address();
        $licence = isset($posted['licence']) ? $this->security->xss_clean(trim($posted['licence'])) : null;
        $token = isset($posted['token']) ? $this->security->xss_clean(trim($posted['token'])) : null;

        if (empty($nif)) {
            $this->output->set_status_header(400)->set_output(json_encode(['status' => 'error', 'message' => 'Falta TaxRegistrationNumber']));
            return;
        }

        // Validação do token HMAC (opcional mas recomendado)
        if ($this->sif_require_token) {
            $provided = $token;
            // Calcular expected com os campos que o cliente conhece
            $payload = $nif . '|' . (string)$logged_status . '|' . ($licence ?: '');
            $expected = hash_hmac($this->sif_algo, $payload, $this->sif_secret);
            if (empty($provided) || !hash_equals($expected, $provided)) {
                // Registar tentativa falhada
                $this->load->helper('file');
                $log = date('Y-m-d H:i:s') . " - invalid_token NIF={$nif} IP=" . $this->input->ip_address() . " payload={$payload} provided=" . substr($provided,0,40) . "\n";
                @write_file(APPPATH . '../logs/sif_api_tokens.log', $log, 'a');
                $this->output->set_status_header(403)->set_output(json_encode(['status' => 'error', 'message' => 'Token inválido']));
                return;
            }
        }

        // Opcional: validar token/secret aqui (recomendado)

        $data = [
            'TaxRegistrationNumber' => $nif,
            'licence' => $licence,
            'logged_status' => $logged_status,
            'logged_ip' => $logged_ip,
            'created_at' => date('Y-m-d H:i:s')
        ];

        $ok = $this->logged_model->insert($data);

        if ($ok) {
            $last = $this->logged_model->get_last_by_nif($nif);
            // Adiciona lookup na tabela tec_licences (último registo por NIF)
            $lic_record = $this->logged_model->get_latest_licence_by_nif($nif);
            $this->output->set_output(json_encode(['status' => 'ok', 'message' => 'registered', 'data' => $last, 'licence_record' => $lic_record]));
        } else {
            $this->output->set_status_header(500)->set_output(json_encode(['status' => 'error', 'message' => 'Erro ao registar']));
        }
    }

    /**
     * Retorna o último registo de licença por NIF.
     * Compatível com /api/get_license_by_nif (POST ou GET).
     */
    public function get_license_by_nif()
    {
        // Aceita POST ou GET
        $method = $this->input->method();
        $posted = [];
        if ($method === 'post') {
            $raw = trim(file_get_contents('php://input'));
            if (!empty($raw) && $this->input->server('CONTENT_TYPE') && strpos($this->input->server('CONTENT_TYPE'), 'application/json') !== false) {
                $posted = json_decode($raw, true);
                if (!is_array($posted)) $posted = [];
            } else {
                $posted = $this->input->post();
            }
        } else {
            $posted = $this->input->get();
        }

        $nif = isset($posted['TaxRegistrationNumber']) ? $this->security->xss_clean(trim($posted['TaxRegistrationNumber'])) : null;
        $token = isset($posted['token']) ? $this->security->xss_clean(trim($posted['token'])) : null;
        $ts = isset($posted['ts']) ? trim($posted['ts']) : null;
        $nonce = isset($posted['nonce']) ? trim($posted['nonce']) : null;

        if (empty($nif)) {
            $this->output->set_status_header(400)->set_output(json_encode(['status' => 'error', 'message' => 'Falta TaxRegistrationNumber']));
            return;
        }

        // validar token (payload: nif) + optional ts/nonce replay protection
        if ($this->sif_require_token) {
            if ($this->config->item('sif_api_require_nonce', 'sif_api')) {
                if (empty($ts) || empty($nonce)) {
                    $this->output->set_status_header(400)->set_output(json_encode(['status' => 'error', 'message' => 'Falta ts ou nonce']));
                    return;
                }
                // validar janela temporal
                $window = (int)$this->config->item('sif_api_ts_window', 'sif_api');
                if (!ctype_digit((string)$ts)) {
                    $this->output->set_status_header(400)->set_output(json_encode(['status' => 'error', 'message' => 'ts invalido']));
                    return;
                }
                $now = time();
                $ts_int = (int)$ts;
                if (abs($now - $ts_int) > $window) {
                    $this->output->set_status_header(403)->set_output(json_encode(['status' => 'error', 'message' => 'Timestamp fora da janela']));
                    return;
                }
                // verificar nonce não usado
                if ($this->logged_model->nonce_exists($nonce)) {
                    $this->output->set_status_header(403)->set_output(json_encode(['status' => 'error', 'message' => 'Nonce já usado']));
                    return;
                }
                $payload = $nif . '|' . $ts . '|' . $nonce;
                $expected = hash_hmac($this->sif_algo, $payload, $this->sif_secret);
                if (empty($token) || !hash_equals($expected, $token)) {
                    $this->output->set_status_header(403)->set_output(json_encode(['status' => 'error', 'message' => 'Token inválido']));
                    return;
                }
                // store nonce to prevent reuse
                $this->logged_model->store_nonce($nonce);
            } else {
                $expected = hash_hmac($this->sif_algo, $nif, $this->sif_secret);
                if (empty($token) || !hash_equals($expected, $token)) {
                    $this->output->set_status_header(403)->set_output(json_encode(['status' => 'error', 'message' => 'Token inválido']));
                    return;
                }
            }
        }

        $lic = $this->logged_model->get_latest_licence_by_nif($nif);
        if (!$lic) {
            $this->output->set_status_header(404)->set_output(json_encode(['status' => 'error', 'message' => 'No licence found']));
            return;
        }

        $this->output->set_output(json_encode(['status' => 'ok', 'data' => $lic]));
    }

}
