<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

class Sales extends MY_Controller {

	function __construct()
	{
		parent::__construct();

		if (!$this->loggedIn) {
			redirect('login');
		}
		$this->load->library('form_validation');
		$this->load->model('sales_model');
		$this->digital_file_types = 'zip|pdf|doc|docx|xls|xlsx|jpg|png|gif';
        $this->load->helper("database_helper");
	}

	/* =========================================================
	 *  AGT: gerar InvoiceNo via tec_numbering + tec_settings
	 *  Regra:
	 *   - Se já existe tec_numbering (mesmo InvoiceType/InvoiceYear) -> continua number+1 e usa mesma agt_series_code
	 *   - Se não existe -> busca série em tec_settings.<TIPO>-seriesCode e começa /1
	 *   - Se não houver série -> NÃO cria
	 *
	 *  InvoiceNo final: "{InvoiceType} {SERIE}/{NUM}"
	 *  Ex: FT FT7825S1708N/2
	 * =======================================================*/
	private function agt_build_invoice_no($invoiceType, $invoiceYear)
	{
		$date_now = date('Y-m-d H:i:s');

		$this->db->trans_begin();

		// bloquear a última linha para evitar duplicar número em concorrência
		$qLast = $this->db->query(
			"SELECT id, number, agt_series_code
			   FROM tec_numbering
			  WHERE InvoiceYear = ?
				AND InvoiceType = ?
		   ORDER BY id DESC
			  LIMIT 1
			  FOR UPDATE",
			array($invoiceYear, $invoiceType)
		);
		if (!$qLast) {
			$err = $this->db->error();
			$this->db->trans_rollback();
			return array('ok'=>false, 'error'=>'Falha na query tec_numbering: ' . (isset($err['message'])?$err['message']:''));
		}
		$last = $qLast->row_array();

		$invoice_number  = 1;
		$agt_series_code = '';

		if (!empty($last)) {
			$invoice_number  = ((int)$last['number']) + 1;
			$agt_series_code = isset($last['agt_series_code']) ? trim((string)$last['agt_series_code']) : '';
		}

		// se não tinha série, buscar em tec_settings.<TIPO>-seriesCode
		if ($agt_series_code === '') {
			$col = $invoiceType . '-seriesCode'; // ex: FT-seriesCode

			$qSet = $this->db->select("`{$col}` AS sc", false)
				->from('tec_settings')
				->limit(1)
				->get();
			if (!$qSet) {
				$err = $this->db->error();
				$this->db->trans_rollback();
				return array('ok'=>false, 'error'=>'Falha na query tec_settings: ' . (isset($err['message'])?$err['message']:''));
			}
			$set = $qSet->row_array();

			$agt_series_code = isset($set['sc']) ? trim((string)$set['sc']) : '';
		}

		// sem série: aborta
		if ($agt_series_code === '') {
			$this->db->trans_rollback();
			return array('ok'=>false, 'error'=>"Sem série para {$invoiceType}. Configure em tec_settings ({$invoiceType}-seriesCode).");
		}

		// grava 1 linha de histórico/controlo (compatível com o POS)
		$this->db->insert('tec_numbering', array(
			'number'          => $invoice_number,
			'InvoiceType'     => $invoiceType,
			'InvoiceYear'     => $invoiceYear,
			'agt_series_code' => $agt_series_code,
			'agt_updated_at'  => $date_now
		));

		if ($this->db->trans_status() === FALSE) {
			$this->db->trans_rollback();
			return array('ok'=>false, 'error'=>'Falha ao gerar numeração AGT.');
		}

		$this->db->trans_commit();

		$invoice_no = $invoiceType . " " . $agt_series_code . "/" . $invoice_number;

		return array(
			'ok'         => true,
			'invoice_no' => $invoice_no,
			'next_no'    => $invoice_number,
			'series'     => $agt_series_code
		);
	}
	/* ========================================================= */


	function index()
	{
		$this->data['error'] = (validation_errors()) ? validation_errors() : $this->session->flashdata('error');
		$this->data['page_title'] = lang('sales');
		$bc = array(array('link' => '#', 'page' => lang('sales')));
		$meta = array('page_title' => lang('sales'), 'bc' => $bc);
		$this->page_construct('sales/index', $this->data, $meta);
	}

	function invoice_lists()
	{
		$this->data['error'] = (validation_errors()) ? validation_errors() : $this->session->flashdata('error');
		$this->data['page_title'] = lang('invoice_lists');
		$bc = array(array('link' => '#', 'page' => lang('invoice_lists')));
		$meta = array('page_title' => lang('invoice_lists'), 'bc' => $bc);
		$this->page_construct('sales/invoice_lists', $this->data, $meta);
	}

	function pro_forma()
	{
		$this->data['error'] = (validation_errors()) ? validation_errors() : $this->session->flashdata('error');
		$this->data['page_title'] = lang('invoice_lists');
		$bc = array(array('link' => '#', 'page' => lang('invoice_lists')));
		$meta = array('page_title' => lang('invoice_lists'), 'bc' => $bc);
		$this->page_construct('sales/pro_forma', $this->data, $meta);
	}

	function convert_invoice($id){
		$sale_info = get_row("tec_sales",array("id"=>$id));

		if (!$sale_info) {
			$this->session->set_flashdata('error', 'Factura não encontrada');
			redirect("sales/pro_forma");
			return;
		}

		// Verifica se já é FT
		if (isset($sale_info['InvoiceType']) && $sale_info['InvoiceType'] === 'FT') {
			$this->session->set_flashdata('error', 'Esta factura já é FT');
			redirect("sales/pro_forma");
			return;
		}

		// =========================================================
		// AGT: gerar InvoiceNo FT a partir da série/ordem
		// =========================================================
		$doc = $this->agt_build_invoice_no("FT", date("Y"));
		if (empty($doc['ok'])) {
			$this->session->set_flashdata('error', $doc['error']);
			redirect("sales/pro_forma");
			return;
		}

		$invoice_number = $doc['next_no'];
		$invoice_no = $doc['invoice_no'];
		$agt_series_code = $doc['series'];

		$date = date('Y-m-d H:i:s');

		// Busca hash anterior (última FT)
		$before_hash = '';
		$last_sale = $this->db->select('Hash')
			->from('tec_sales')
			->where('id <', $id)
			->order_by('id', 'DESC')
			->limit(1)
			->get()
			->row();
		if ($last_sale && isset($last_sale->Hash)) {
			$before_hash = substr($last_sale->Hash, 0, 4);
		}

		// Gera hash
		$hash_str = date("Y-m-d",strtotime($date)).";"
			.date("Y-m-d",strtotime($date))."T".date("H:i:s",strtotime($date)).";"
			.$invoice_no.";".str_replace(",","",number_format($sale_info['grand_total'],2)).";"
			.$before_hash;

		$hash_no = $hash_str;

		$sign_key = get_row("tec_signkey", array("id"=>1));
		if (!$sign_key || !isset($sign_key['private'])) {
			$this->session->set_flashdata('error', 'Chave privada não encontrada');
			redirect("sales/pro_forma");
			return;
		}

		$privateKey = $sign_key['private'];
		openssl_sign($hash_str, $crypttext, $privateKey, "sha1");
		$hash = base64_encode($crypttext);

		// =========================================================
		// Cria nova venda FT (duplica dados da FP)
		// =========================================================
		$new_sale = $sale_info;
		unset($new_sale['id']); // Remove ID para criar novo
		
		$new_sale['InvoiceType'] = 'FT';
		$new_sale['InvoiceNo'] = $invoice_no;
		$new_sale['InvoiceYear'] = date("Y");
		$new_sale['Hash'] = $hash;
		$new_sale['HashControl'] = '1';
		$new_sale['date'] = $date;
		$new_sale['InvoiceStatusDate'] = date("Y-m-d", strtotime($date)) . "T" . date("H:i:s", strtotime($date));
		$new_sale['OriginatingON'] = $sale_info['InvoiceNo']; // Referencia a FP original
		$new_sale['mesa'] = 'N';
		$new_sale['status'] = 'no paid';
		$new_sale['paid'] = 0;
		
		// Remove variações de hash_no (case sensitive)
		unset($new_sale['Hash_no'], $new_sale['hash_no']);
		// Adiciona a versão correta
		$new_sale['hash_no'] = $hash_no;
		
		// Limpa campos AGT (nova factura não tem estado AGT ainda)
		$new_sale['agt_status'] = null;
		$new_sale['agt_request_id'] = null;
		$new_sale['agt_submitted_at'] = null;
		$new_sale['agt_error_json'] = null;
		$new_sale['agt_last_status'] = null;
		
		// Remove campos que podem causar conflito
		unset($new_sale['updated_at'], $new_sale['updated_by']);
		
		// Remove campos vazios/null que podem causar problemas
		foreach ($new_sale as $key => $value) {
			if ($value === '' || $value === null) {
				unset($new_sale[$key]);
			}
		}

		$this->db->insert('tec_sales', $new_sale);
		$new_sale_id = $this->db->insert_id();

		if (!$new_sale_id) {
			// Log do erro SQL
			$db_error = $this->db->error();
			$error_msg = 'Erro ao criar factura FT';
			if (!empty($db_error['message'])) {
				$error_msg .= ': ' . $db_error['message'];
			}
			
			// Grava log em ficheiro para debug
			@file_put_contents(
				APPPATH . 'logs/convert_fp_error_' . date('Y-m-d') . '.log',
				date('Y-m-d H:i:s') . " - FP ID: {$id}\n" .
				"DB Error: " . print_r($db_error, true) . "\n" .
				"Data: " . print_r($new_sale, true) . "\n\n",
				FILE_APPEND
			);
			
			$this->session->set_flashdata('error', $error_msg);
			redirect("sales/pro_forma");
			return;
		}

		// =========================================================
		// Copia itens da FP para a FT
		// =========================================================
		$fp_items = $this->db->get_where('tec_sale_items', array('sale_id'=>$id))->result_array();
		foreach ($fp_items as $item) {
			$new_item = $item;
			unset($new_item['id']);
			$new_item['sale_id'] = $new_sale_id;
			$this->db->insert('tec_sale_items', $new_item);
		}

		// =========================================================
		// Marca a FP original como convertida
		// =========================================================
		$this->db->where('id', $id);
		$this->db->update('tec_sales', array(
			'mesa' => 'CONVERTIDA',
			'note' => (isset($sale_info['note']) ? $sale_info['note'] . ' | ' : '') . 'Convertida para FT: ' . $invoice_no
		));

		// =========================================================
		// Envia para AGT automaticamente (em background)
		// =========================================================
		$base_url = base_url();
		
		// 1. Envia para AGT
		$agt_url = $base_url . 'index.php/Agt/registar_factura/' . $new_sale_id . '?send=1';
		@file_get_contents($agt_url, false, stream_context_create(array(
			'http' => array(
				'timeout' => 60,
				'ignore_errors' => true
			)
		)));
		
		// 2. Aguarda 10 segundos
		sleep(10);
		
		// 3. Consulta estado
		$estado_url = $base_url . 'index.php/Agt/obter_estado/' . $new_sale_id . '?send=1';
		@file_get_contents($estado_url, false, stream_context_create(array(
			'http' => array(
				'timeout' => 30,
				'ignore_errors' => true
			)
		)));

		$this->session->set_flashdata('message', 'Factura Proforma convertida para FT: ' . $invoice_no . ' e enviada para AGT');
		$this->session->set_userdata('reprint', $new_sale_id);
		redirect("sales/invoice_lists");
	}

	function convert_Nc_invoice(){
		$id = $this->input->get("id");
		$reason = $this->input->get("reason");
		$sale_info = get_row("tec_sales",array("id"=>$id));
		$data = array('mesa' => "NC", );

		$this->db->where(array('sale_id' => $id));
		$sale_items = $this->db->get("sale_items")->result_array();
		foreach ($sale_items as $key => $sale_item) {
			$product = $this->db->get_where("products",array("id"=>$sale_item['product_id']))->row_array();
			$quantity =  $product['quantity'] + $sale_item['quantity'];
			$this->db->update('products', array('quantity' => $quantity),  array('id' => $sale_item['product_id']));
		}

		update_row("tec_sales",$data,array("id"=>$id));

		// =========================================================
		// AGT: gerar InvoiceNo NC via série/ordem
		// =========================================================
		$doc = $this->agt_build_invoice_no("NC", date("Y"));
		if (empty($doc['ok'])) {
			echo json_encode(array("status"=>"error","message"=>$doc['error']));
			return;
		}

		$invoice_number = $doc['next_no'];
		$invoice_no = $doc['invoice_no'];

		// Hash anterior (última NC do ano)
		$before_hash = "";
		$sale_data = get_row("tec_sales", array("InvoiceYear"=>date("Y"), "InvoiceType"=>"NC"), "id DESC");
		if (!empty($sale_data) && isset($sale_data['Hash'])) {
			$before_hash = $sale_data['Hash'];
		}

		$date = date('Y-m-d H:i:s');

		$hash_str = date("Y-m-d",strtotime($date)).";".date("Y-m-d",strtotime($date))."T".date("H:i:s",strtotime($date)).";".$invoice_no.";".str_replace(",","",number_format($sale_info['grand_total'],2)).";".$before_hash;
		$hash_no = $hash_str;

		$sign_key = get_row("tec_signkey",array("id"=>1));
		$privateKey=$sign_key['private'];
		$publickey=$sign_key['public'];

		openssl_sign ($hash_str, $crypttext, $privateKey, "sha1");
		$hash_str = base64_encode($crypttext);

		$data = $sale_info;
		unset($data['id']);
		$data['OriginatingON'] = $sale_info['InvoiceNo'];
		$data['Hash'] = $hash_str;
		$data['Hash_no'] = $hash_no;
		$data['InvoiceStatusDate'] = date("Y-m-d", strtotime($date)) . "T" . date("H:i:s", strtotime($date));
		$data['date'] = $date;
		$data['InvoiceType'] = "NC";
		$data['InvoiceNo'] = $invoice_no;
		$data['Reason'] = $reason;

		foreach ($data as $key => $value) {
			if(is_null($value) || $value == "") unset($data[$key]);
		}
		unset($data['updated_at']);

		create_row("tec_sales", $data);
		$new_sale_info = get_row("tec_sales",array("InvoiceNo" => $invoice_no));
		$sale_id = $new_sale_info['id'];

		$sale_item_infos = get_rows("tec_sale_items",array("sale_id"=>$sale_info['id']));
		foreach ($sale_item_infos as $key => $sale_item_info) {
			unset($sale_item_info['id']);
			$sale_item_info['sale_id'] = $sale_id;
			create_row("tec_sale_items", $sale_item_info);
		}
		echo json_encode(array("status"=>"ok","sale_id" => $sale_id));
	}

	function edit_invoice($eid = NULL){
		$this->load->library('form_validation');
		$this->load->model('pos_model');
		$this->load->helper('database_helper');

		if (isset($eid) && !empty($eid)) {
			$sale = $this->pos_model->getSaleByID($eid);
			$inv_items = $this->pos_model->getAllSaleItems($eid);
			krsort($inv_items);
			$c = rand(100000, 9999999);
			foreach ($inv_items as $item) {
				$row = $this->site->getProductByID($item->product_id);
				if (!$row) {
					$row = json_decode('{}');
				}
				$row->price = $item->net_unit_price;
				$row->unit_price = $item->unit_price;
				$row->real_unit_price = $item->real_unit_price;
				$row->discount = $item->discount;
				$row->qty = $item->quantity;
				$combo_items = FALSE;
				$row->quantity += $item->quantity;
				if ($row->type == 'combo') {
					$combo_items = $this->pos_model->getComboItemsByPID($row->id);
					foreach ($combo_items as $combo_item) {
						$combo_item->quantity += ($combo_item->qty * $item->quantity);
					}
				}
				$ri = $this->Settings->item_addition ? $row->id : $c;
				$pr[$ri] = array('id' => $c, 'item_id' => $row->id, 'label' => $row->name . " (" . $row->code . ")", 'row' => $row, 'combo_items' => $combo_items);
				$c++;
			}
			$this->data['items'] = json_encode($pr);
			$this->data['eid'] = $eid;
			$this->data['sale'] = $sale;
			$this->data['message'] = lang('sale_loaded');
		}

		$this->data['error'] = (validation_errors()) ? validation_errors() : $this->session->flashdata('error');
		$this->data['eid'] = isset($eid) ? $eid : 0;
		$this->data['customers'] = $this->site->getAllCustomers();

		$this->data["tcp"] = $this->pos_model->products_count($this->Settings->default_category);

		// $this->data['products'] = $this->ajaxproducts($this->Settings->default_category, 1);
		$this->data['categories'] = $this->site->getAllCategories();
		$this->data['message'] = $this->session->flashdata('message');
		$this->data['suspended_sales'] = $this->site->getUserSuspenedSales();
		$this->data['page_title'] = lang('pos');
		$bc = array(array('link' => '#', 'page' => lang('pos')));
		$meta = array('page_title' => lang('pos'), 'bc' => $bc);
		$this->page_construct('sales/edit_invoice', $this->data, $meta);

	}

	function update_invoice(){
		$this->load->model("pos_model");
		$eid = $this->input->post("eid");
		$quantity = "quantity";
		$product = "product";
		$unit_cost = "unit_cost";
		$tax_rate = "tax_rate";

		$date = date('Y-m-d H:i:s');
		$customer_id = $this->input->post('customer_id');
		$customer_details = $this->pos_model->getCustomerByID($customer_id);
		$customer = $customer_details->name;
		$note = $this->tec->clear_tags($this->input->post('spos_note'));

		$total = 0;
		$product_tax = 0;
		$order_tax = 0;
		$product_discount = 0;
		$order_discount = 0;
		$percentage = '%';
		$i = isset($_POST['product_id']) ? sizeof($_POST['product_id']) : 0;
		for ($r = 0; $r < $i; $r++) {
			$item_id = $_POST['product_id'][$r];
			$real_unit_price = $this->tec->formatDecimal($_POST['real_unit_price'][$r]);
			$item_quantity = $_POST['quantity'][$r];
			$item_discount = isset($_POST['product_discount'][$r]) ? $_POST['product_discount'][$r] : '0';

			if (isset($item_id) && isset($real_unit_price) && isset($item_quantity)) {
				$product_details = $this->site->getProductByID($item_id);
				$unit_price = $real_unit_price;

				$pr_discount = 0;
				if (isset($item_discount)) {
					$discount = $item_discount;
					$dpos = strpos($discount, $percentage);
					if ($dpos !== false) {
						$pds = explode("%", $discount);
						$pr_discount = (($this->tec->formatDecimal($unit_price)) * (Float) ($pds[0])) / 100;
					} else {
						$pr_discount = $this->tec->formatDecimal($discount);
					}
				}
				$unit_price = $this->tec->formatDecimal($unit_price - $pr_discount);
				$item_net_price = $unit_price;
				$pr_item_discount = $this->tec->formatDecimal($pr_discount * $item_quantity);
				$product_discount += $pr_item_discount;

				$pr_item_tax = 0;
				$item_tax = 0;
				$tax = 0;
				if (isset($product_details->tax) && $product_details->tax != 0) {

					if ($product_details && $product_details->tax_method == 1) {
						$item_tax = $this->tec->formatDecimal((($unit_price) * $product_details->tax) / 100);
						$tax = $product_details->tax . "%";
						// Calcular imposto total sem arredondamento intermediário e depois aplicar ceil
						$taxValue = ($unit_price * $product_details->tax / 100) * $item_quantity;
						$pr_item_tax = ceil($taxValue * 100) / 100;
					} else {
						$item_tax = $this->tec->formatDecimal((($unit_price) * $product_details->tax) / (100 + $product_details->tax));
						$tax = $product_details->tax . "%";
						// Calcular imposto total sem arredondamento intermediário e depois aplicar ceil
						$taxValue = (($unit_price * $product_details->tax) / (100 + $product_details->tax)) * $item_quantity;
						$pr_item_tax = ceil($taxValue * 100) / 100;
						/* $item_net_price -= $item_tax; */
					}
				}

				$product_tax += $pr_item_tax;
				$subtotal = (($item_net_price * $item_quantity) + $pr_item_tax);

				$products[] = array(
					'product_id' => $item_id,
					'quantity' => $item_quantity,
					'unit_price' => $unit_price,
					'net_unit_price' => $item_net_price,
					'discount' => $item_discount,
					'item_discount' => $pr_item_discount,
					'tax' => str_replace("%", "", $tax),
					'item_tax' => $pr_item_tax,
					'subtotal' => $subtotal,
					'real_unit_price' => $real_unit_price,
					'cost' => $product_details->cost,
					'tax_id'=>$product_details->tax_id
				);

				$total += $item_net_price * $item_quantity;
			}
		}

		if ($this->input->post('order_discount')) {
			$order_discount_id = $this->input->post('order_discount');
			$opos = strpos($order_discount_id, $percentage);
			if ($opos !== false) {
				$ods = explode("%", $order_discount_id);
				$order_discount = $this->tec->formatDecimal((($total + $product_tax) * (Float) ($ods[0])) / 100);
			} else {
				$order_discount = $this->tec->formatDecimal($order_discount_id);
			}
		} else {
			$order_discount_id = NULL;
		}
		$total_discount = $this->tec->formatDecimal($order_discount + $product_discount);

		if ($this->input->post('order_tax')) {
			$order_tax_id = $this->input->post('order_tax');
			$opos = strpos($order_tax_id, $percentage);
			if ($opos !== false) {
				$ots = explode("%", $order_tax_id);
				$order_tax = $this->tec->formatDecimal((($total + $product_tax - $order_discount) * (Float) ($ots[0])) / 100);
			} else {
				$order_tax = $this->tec->formatDecimal($order_tax_id);
			}
		} else {
			$order_tax_id = NULL;
			$order_tax = 0;
		}

		$total_tax = $this->tec->formatDecimal($product_tax + $order_tax);
		$grand_total = $this->tec->formatDecimal($this->tec->formatDecimal($total) + $total_tax - $order_discount);

		$_POST['amount'] = $grand_total;
		$paid = $this->input->post('amount') ? $this->input->post('amount') : 0;
		$status = 'paid';
		if($this->input->post("invoice_type") == "FT") {
			$paid = 0;
			$status = 'no paid';
		}

		$round_total = $this->tec->roundNumber($grand_total, $this->Settings->rounding);
		$rounding = $this->tec->formatMoney($round_total - $grand_total);
		$balance = empty($this->input->post('balance_amount')) ? 0 : $this->tec->formatDecimal($this->input->post('balance_amount'));

		$data = array('date' => $date,
			'total' => $this->tec->formatDecimal($total),
			'product_discount' => $this->tec->formatDecimal($product_discount),
			'order_discount_id' => $order_discount_id,
			'order_discount' => $order_discount,
			'total_discount' => $total_discount,
			'total_balance' => $balance,
			'product_tax' => $this->tec->formatDecimal($product_tax),
			'order_tax_id' => $order_tax_id,
			'order_tax' => $order_tax,
			'total_tax' => $total_tax,
			'grand_total' => $grand_total,
			'total_items' => $this->input->post('total_items'),
			'total_quantity' => $this->input->post('total_quantity'),
			'rounding' => $rounding,
			'paid' => $paid,
			'status' => $status,
			'created_by' => $this->session->userdata('user_id'),
			'note' => $note,
			'InvoiceYear' => date("Y"),
			'InvoiceStatusDate' => date("Y-m-d", strtotime($date)) . "T" . date("H:i:s", strtotime($date)),
			'SourceBilling' => "P",
			'CashVATSchemeIndicator' => 1,
			'ThirdPartiesBillingIndicator' => 0,
			'SelfBillingIndicator' => 0,
			'Reason' => $this->input->post("Reason")
		);

		$sale_info = get_row("tec_sales",array("id"=>$eid));
		unset($data['date'], $data['InvoiceNo']);
		if($sale_info['InvoiceType'] != "FR") unset($data['paid']);
		$data['updated_at'] = date('Y-m-d H:i:s');
		$data['updated_by'] = $this->session->userdata('user_id');

		$amount =  $paid;
		$payment = array(
			'date' => $date,
			'amount' => $amount,
			'pos_paid' => $this->tec->formatDecimal($this->input->post('amount')),
			'pos_balance' => $this->tec->formatDecimal($this->input->post('balance_amount')),
		);

		if($sale_info['InvoiceType'] == "FR"){
			$row = get_row("tec_payments",array("sale_id"=>$eid));
			update_row("tec_payments", $payment, array("sale_id"=>$eid));
		}

		if ($this->pos_model->updateSale($eid, $data, $products)) {
			$this->session->set_userdata('rmspos', 1);
			$this->session->set_flashdata('message', lang("sale_updated"));
			$this->session->set_userdata('reprint', $eid);
			redirect("sales/invoice_lists");
		} else {
			$this->session->set_flashdata('error', lang("action_failed"));
			redirect("pos/?edit=" . $eid);
		}
	}

	function get_pro_forma()
	{
		$this->load->library('datatables');
		$this->datatables->select("id, date, customer_name, InvoiceNo, grand_total, InvoiceStatus");
		$this->datatables->from('sales');
		$this->datatables->where('InvoiceType', "FP");
		$this->datatables->where('mesa', 'N'); // Só FP não convertidas

		if(!$this->Admin) {
			$user_id = $this->session->userdata('user_id');
			$this->datatables->where('created_by', $user_id);
		}

		$this->datatables->add_column("Actions", "<div class='text-center'><div class='btn-group'><a href='#' onClick=\"MyWindow=window.open('" . site_url('pos/printInvoice/$1') . "',target='_blank'); return false;\" title='".lang("print_invoice")."' class='tip btn btn-success btn-xs print$1'><i class='fa fa-print'></i></a> <a href='#' onClick=\"MyWindow=window.open('" . site_url('pos/view/$1/1') . "', 'MyWindow','toolbar=no,location=no,directories=no,status=no,menubar=yes,scrollbars=yes,resizable=yes,width=350,height=600'); return false;\" title='".lang("view_invoice")."' class='tip btn btn-primary btn-xs'><i class='fa fa-list'></i></a> <a href='".site_url('sales/convert_invoice/$1')."' title='" . lang("convert invoice") . "' class='tip btn btn-primary btn-xs' ><i class='fa fa-exchange'></i></a> <a href='" . site_url('sales/edit_invoice/$1') . "' title='".lang("edit_invoice")."' class='tip btn btn-warning btn-xs'><i class='fa fa-edit'></i></a> <a href='" . site_url('sales/delete/$1') . "' onClick=\"return confirm('". lang('alert_x_sale') ."')\" title='".lang("delete_sale")."' class='tip btn btn-danger btn-xs'><i class='fa fa-trash-o'></i></a></div></div>","id");

		$this->datatables->unset_column('id');
		echo $this->datatables->generate();
	}

	function get_invoices()
	{
		$this->load->library('datatables');

		$this->datatables->select("
			id,
			date,
			customer_name,
			InvoiceNo,
			OriginatingON,
			grand_total,
			paid,
			(grand_total - paid) AS balance,
			mesa,
			agt_status
		");

		$this->datatables->from('sales');
		
		// Lista FT E Notas de Crédito onde OriginatingON começa com FT
		$this->db->start_cache();
		$this->datatables->where_in('InvoiceType', array('FT'));
		$this->datatables->or_where("InvoiceType = 'NC' AND OriginatingON LIKE 'FT%'", NULL, FALSE);
		$this->db->stop_cache();

		if (!$this->Admin) {
			$user_id = $this->session->userdata('user_id');
			$this->datatables->where('created_by', $user_id);
		}

		$this->datatables->add_column(
			"Actions",
			"<div class='text-center'>
				<div class='btn-group'>
					<a href='#' onClick=\"MyWindow=window.open('" . site_url('pos/printInvoice/$1') . "',target='_blank'); return false;\" class='tip btn btn-warning btn-xs print$1' title='".lang("print_invoice")."'>
						<i class='fa fa-print'></i>
					</a>

					<a href='" . site_url('sales/add_payment/$1') . "' class='add_payment_btn tip btn btn-success btn-xs' data-toggle='ajax' title='".lang("add_payment")."'>
						<i class='fa fa-briefcase'></i>
					</a>

					<a href='#' onClick=\"MyWindow=window.open('" . site_url('pos/guia_transporte/$1') . "',target='_blank'); return false;\" class='tip btn btn-primary btn-xs' title='Guia de Transporte'>
						<i class='fa fa-car'></i>
					</a>

					<a href='" . site_url('sales/payments/$1') . "' class='tip btn btn-primary btn-xs' data-toggle='ajax' title='".lang("view_payments")."'>
						<i class='fa fa-money'></i>
					</a>

					<a title='" . lang("Nota de Crédito") . "'
					   class='tip btn btn-danger btn-xs convert-nc-btn'
					   data-id='$1'>
						<i class='fa fa-exchange'></i>
					</a>

					<a href='" . site_url('Agt/registar_factura/$1') . "?send=1' class='tip btn btn-info btn-xs' title='Enviar AGT'>
						<i class='fa fa-cloud-upload'></i>
					</a>

					<a href='" . site_url('Agt/obter_estado/$1?send=1') . "' class='tip btn btn-default btn-xs' title='Actualizar estado AGT'>
						<i class='fa fa-refresh'></i>
					</a>
				</div>
			</div>",
			"id"
		);

		$this->datatables->unset_column('id');
		echo $this->datatables->generate();
	}

	function get_sales()
	{
		$this->load->library('datatables');

		$this->datatables->select("
			id,
			date,
			customer_name,
			InvoiceNo,
			total_discount,
			grand_total,
			paid,
			mesa,
			agt_status
		");

		$this->datatables->from('tec_sales');
		
		// Lista facturas FR/AF E Notas de Crédito onde OriginatingON começa com FR
		$this->db->start_cache();
		$this->datatables->where_in('InvoiceType', array('FR', 'AF'));
		$this->datatables->or_where("InvoiceType = 'NC' AND OriginatingON LIKE 'FR%'", NULL, FALSE);
		$this->db->stop_cache();

		if (!$this->Admin) {
			$user_id = $this->session->userdata('user_id');
			$this->datatables->where('created_by', $user_id);
		}

		$this->datatables->add_column(
			"Actions",
			"
			<div class='text-center'>
				<div class='btn-group'>
					<a href='#'
					   onClick=\"MyWindow=window.open('" . site_url('pos/printInvoice/$1') . "',target='_blank'); return false;\"
					   title='" . lang("print_invoice") . "'
					   class='tip btn btn-warning btn-xs print$1'>
						<i class='fa fa-print'></i>
					</a>

					<a href='#'
					   onClick=\"MyWindow=window.open('" . site_url('pos/view/$1/1') . "',target='_blank'); return false;\"
					   title='" . lang("Imprimir POS") . "'
					   class='tip btn btn-info btn-xs'>
						<i class='fa fa-print'></i>
					</a>

					<a onClick=\"MyWindow=window.open('" . site_url('pos/guia_transporte/$1') . "',target='_blank');\"
					   title='" . lang("Guia de Transporte") . "'
					   class='tip btn btn-primary btn-xs'
					   data-toggle='ajax'>
						<i class='fa fa-car'></i>
					</a>

					<a href='" . site_url('sales/payments/$1') . "'
					   title='" . lang("view_payments") . "'
					   class='tip btn btn-primary btn-xs'
					   data-toggle='ajax'>
						<i class='fa fa-money'></i>
					</a>

					<a href='" . site_url('sales/add_payment/$1') . "'
					   title='" . lang("add_payment") . "'
					   class='add_payment_btn tip btn btn-success btn-xs'
					   data-toggle='ajax'>
						<i class='fa fa-briefcase'></i>
					</a>

					<a title='" . lang("Nota de Crédito") . "'
					   class='tip btn btn-danger btn-xs convert-nc-btn'
					   data-id='$1'>
						<i class='fa fa-exchange'></i>
					</a>
					
				</div>
			</div>
			",
			"id"
		);

		$this->datatables->unset_column('id');
		echo $this->datatables->generate();
	}

	function opened()
	{
		$this->data['error'] = (validation_errors()) ? validation_errors() : $this->session->flashdata('error');
		$this->data['page_title'] = lang('opened_bills');
		$bc = array(array('link' => '#', 'page' => lang('opened_bills')));
		$meta = array('page_title' => lang('opened_bills'), 'bc' => $bc);
		$this->page_construct('sales/opened', $this->data, $meta);
	}

	function get_opened_list()
	{
		$this->load->library('datatables');
		$this->datatables
		->select("id, date, created_by, hold_ref,  CONCAT(total_items, ' (', total_quantity, ')') as items, grand_total", FALSE)
		->from('suspended_sales');

		if(!$this->Staff & Admin)  {
			$user_id = $this->session->userdata('user_id');
			$this->datatables->where('created_by', $user_id);
		}

		$this->datatables->add_column("Actions",
			"<div class='text-center'><div class='btn-group'><a href='" . site_url('pos/?hold=$1') . "' title='".lang("click_to_add")."' class='tip btn btn-info btn-xs'><i class='fa fa-th-large'></i></a>
			<a href='" . site_url('sales/delete_holded/$1') . "' onClick=\"return confirm('". lang('alert_x_holded') ."')\" title='".lang("delete_sale")."' class='tip btn btn-danger btn-xs'><i class='fa fa-trash-o'></i></a></div></div>", "id")
		->unset_column('id');

		echo $this->datatables->generate();
	}

	function delete($id = NULL)
	{
		if(DEMO) {
			$this->session->set_flashdata('error', lang('disabled_in_demo'));
			redirect(isset($_SERVER["HTTP_REFERER"]) ? $_SERVER["HTTP_REFERER"] : 'welcome');
		}

		if($this->input->get('id')){ $id = $this->input->get('id'); }

		if (!$this->Admin) {
			$this->session->set_flashdata('error', lang("access_denied"));
			redirect('sales');
		}

		if ( $this->sales_model->deleteInvoice($id) ) {
			$this->session->set_userdata("cancel_payment","yes");
			$this->session->set_userdata("sale_id",$id);

			$this->session->set_flashdata('message', lang("invoice_deleted"));
			redirect('sales/invoice_lists');
		}
	}

	function delete_holded($id = NULL)
	{
		if($this->input->get('id')){ $id = $this->input->get('id'); }

		if (!$this->Admin) {
			$this->session->set_flashdata('error', lang("access_denied"));
			redirect('sales/opened');
		}

		if ( $this->sales_model->deleteOpenedSale($id) ) {
			$this->session->set_flashdata('message', lang("opened_bill_deleted"));
			redirect('sales/opened');
		}
	}

	/* -------------------------------------------------------------------------------- */

	function payments($id = NULL)
	{
		$this->data['payments'] = $this->sales_model->getSalePayments($id);
		$this->load->view($this->theme . 'sales/payments', $this->data);
	}

	function payment_note($id = NULL)
	{
		$payment = $this->sales_model->getPaymentByID($id);
		$inv = $this->sales_model->getSaleByID($payment->sale_id);
		$this->data['customer'] = $this->site->getCompanyByID($inv->customer_id);
		$this->data['inv'] = $inv;
		$this->data['payment'] = $payment;
		$this->data['page_title'] = $this->lang->line("payment_note");

		$this->load->view($this->theme . 'sales/payment_note', $this->data);
	}

	function add_payment($id = NULL, $cid = NULL)
	{
		$this->load->helper('security');
		if ($this->input->get('id')) {
			$id = $this->input->get('id');
		}

		$this->form_validation->set_rules('amount-paid', lang("amount"), 'required');
		$this->form_validation->set_rules('paid_by', lang("paid_by"), 'required');
		$this->form_validation->set_rules('userfile', lang("attachment"), 'xss_clean');

		if ($this->form_validation->run() == true) {
			if ($this->Admin) {
				$date = $this->input->post('date');
			} else {
				$date = date('Y-m-d H:i:s');
			}

			$row = get_row("tec_sales",array("id"=>$id));
			if (empty($row)) {
				$this->session->set_flashdata('error', 'Venda não encontrada para registrar pagamento.');
				$back = isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : 'sales';
				redirect($back);
				return;
			}
			$payment = array(
				'date' => $date,
				'sale_id' => $id,
				'customer_id' => $cid,
				'reference' => $this->input->post('reference'),
				'amount' => $this->input->post('amount-paid'),
				'paid_by' => $this->input->post('paid_by'),
				'cheque_no' => $this->input->post('cheque_no'),
				'gc_no' => $this->input->post('gift_card_no'),
				'cc_no' => $this->input->post('pcc_no'),
				'cc_holder' => $this->input->post('pcc_holder'),
				'cc_month' => $this->input->post('pcc_month'),
				'cc_year' => $this->input->post('pcc_year'),
				'cc_type' => $this->input->post('pcc_type'),
				'note' => $this->input->post('note'),
				'created_by' => $this->session->userdata('user_id'),
				'PaymentStatus' => "N",
				'SourcePayment' => "P",
				'PaymentType' => "RC",
				'OrderReferences' => $row["InvoiceNo"],
			);

			$doc_rg = $this->agt_build_invoice_no("RC", date("Y"));
			if (empty($doc_rg['ok'])) {
				$this->session->set_flashdata('error', $doc_rg['error']);
				redirect($_SERVER["HTTP_REFERER"]);
				return;
			}
			$payment['InvoiceNo'] = $doc_rg['invoice_no'];

			if ($_FILES['userfile']['size'] > 0) {
				$this->load->library('upload');
				$config['upload_path'] = 'files/';
				$config['allowed_types'] = $this->digital_file_types;
				$config['max_size'] = 2048;
				$config['overwrite'] = FALSE;
				$config['encrypt_name'] = TRUE;
				$this->upload->initialize($config);
				if (!$this->upload->do_upload()) {
					$error = $this->upload->display_errors();
					$this->session->set_flashdata('error', $error);
					redirect($_SERVER["HTTP_REFERER"]);
				}
				$photo = $this->upload->file_name;
				$payment['attachment'] = $photo;
			}

		} elseif ($this->input->post('add_payment')) {
			$this->session->set_flashdata('error', validation_errors());
			$this->tec->dd();
		}

		if ($this->form_validation->run() == true && $this->sales_model->addPayment($payment)) {
			$this->session->set_userdata("add_payment","yes");
			$this->session->set_userdata("sale_id",$id);
			$this->session->set_flashdata('message', lang("payment_added"));
			redirect($_SERVER["HTTP_REFERER"]);
		} else {
			$this->data['error'] = (validation_errors() ? validation_errors() : $this->session->flashdata('error'));
			$sale = $this->sales_model->getSaleByID($id);
			$this->data['inv'] = $sale;

			$this->load->view($this->theme . 'sales/add_payment', $this->data);
		}
	}

	function edit_payment($id = NULL, $sid = NULL)
	{
		if (!$this->Admin) {
			$this->session->set_flashdata('error', lang("access_denied"));
			redirect($_SERVER["HTTP_REFERER"]);
		}
		$this->load->helper('security');
		if ($this->input->get('id')) {
			$id = $this->input->get('id');
		}

		$this->form_validation->set_rules('amount-paid', lang("amount"), 'required');
		$this->form_validation->set_rules('paid_by', lang("paid_by"), 'required');
		$this->form_validation->set_rules('userfile', lang("attachment"), 'xss_clean');
		if ($this->form_validation->run() == true) {
			$payment = array(
				'sale_id' => $sid,
				'reference' => $this->input->post('reference'),
				'amount' => $this->input->post('amount-paid'),
				'paid_by' => $this->input->post('paid_by'),
				'cheque_no' => $this->input->post('cheque_no'),
				'gc_no' => $this->input->post('gift_card_no'),
				'cc_no' => $this->input->post('pcc_no'),
				'cc_holder' => $this->input->post('pcc_holder'),
				'cc_month' => $this->input->post('pcc_month'),
				'cc_year' => $this->input->post('pcc_year'),
				'cc_type' => $this->input->post('pcc_type'),
				'note' => $this->input->post('note'),
				'updated_by' => $this->session->userdata('user_id'),
				'updated_at' => date('Y-m-d H:i:s'),
			);

			if ($this->Admin) {
				$payment['date'] = $this->input->post('date');
			}

			if ($_FILES['userfile']['size'] > 0) {
				$this->load->library('upload');
				$config['upload_path'] = 'files/';
				$config['allowed_types'] = $this->digital_file_types;
				$config['max_size'] = 2048;
				$config['overwrite'] = FALSE;
				$config['encrypt_name'] = TRUE;
				$this->upload->initialize($config);
				if (!$this->upload->do_upload()) {
					$error = $this->upload->display_errors();
					$this->session->set_flashdata('error', $error);
					redirect($_SERVER["HTTP_REFERER"]);
				}
				$photo = $this->upload->file_name;
				$payment['attachment'] = $photo;
			}

		} elseif ($this->input->post('edit_payment')) {
			$this->session->set_flashdata('error', validation_errors());
			$this->tec->dd();
		}

		if ($this->form_validation->run() == true && $this->sales_model->updatePayment($id, $payment)) {
			$this->session->set_flashdata('message', lang("payment_updated"));
			redirect("sales");
		} else {

			$this->data['error'] = (validation_errors() ? validation_errors() : $this->session->flashdata('error'));
			$payment = $this->sales_model->getPaymentByID($id);
			if($payment->paid_by != 'cash') {
				$this->session->set_flashdata('error', lang('only_cash_can_be_edited'));
				$this->tec->dd();
			}
			$this->data['payment'] = $payment;
			$this->load->view($this->theme . 'sales/edit_payment', $this->data);
		}
	}

	function delete_payment($id = NULL)
	{
		if($this->input->get('id')){ $id = $this->input->get('id'); }

		if (!$this->Admin) {
			$this->session->set_flashdata('error', lang("access_denied"));
			redirect($_SERVER["HTTP_REFERER"]);
		}

		if ( $this->sales_model->deletePayment($id) ) {
			$this->session->set_flashdata('message', lang("payment_deleted"));
			redirect('sales');
		}
	}

	function printPaymentInvoice($sale_id = NULL){
		$this->load->model("pos_model");

		$this->data['error'] = (validation_errors() ? validation_errors() : $this->session->flashdata('error'));
		$this->data['message'] = $this->session->flashdata('message');
		$inv = $this->pos_model->getSaleByID($sale_id);
		$this->tec->view_rights($inv->created_by);
		$this->load->helper('text');
		$this->data['rows'] = $this->pos_model->getAllSaleItems($sale_id);
		$this->data['customer'] = $this->pos_model->getCustomerByID($inv->customer_id);
		$this->data['inv'] = $inv;
		$this->data['sid'] = $sale_id;
		$this->data['noprint'] = $noprint;
		$this->data['modal'] = false;
		$this->data['payments'] = $this->pos_model->getAllSalePayments($sale_id);
		$this->data['created_by'] = $this->site->getUser($inv->created_by);
		$this->data['page_title'] = lang("invoice");
		$this->session->set_userdata("invoice_status","");

		$this->load->view($this->theme . 'sales/payment_invoice', $this->data);
	}

	function printPaymentInvoiceR($sale_id = NULL){
		$this->load->model("pos_model");

		$get_sales_id = get_row("tec_payments",array("id"=>$sale_id));
		$sale_id2 = $get_sales_id['sale_id'];

		$this->data['error'] = (validation_errors() ? validation_errors() : $this->session->flashdata('error'));
		$this->data['message'] = $this->session->flashdata('message');
		$inv = $this->pos_model->getSaleByID($sale_id2);
		$this->tec->view_rights($inv->created_by);
		$this->load->helper('text');
		$this->data['rows'] = $this->pos_model->getAllSaleItems($sale_id2);
		$this->data['customer'] = $this->pos_model->getCustomerByID($inv->customer_id);
		$this->data['inv'] = $inv;
		$this->data['rec'] = $sale_id;
		$this->data['sid'] = $sale_id2;
		$this->data['noprint'] = $noprint;
		$this->data['modal'] = false;
		$this->data['payments'] = $this->pos_model->getAllSalePaymentsR($sale_id);
		$this->data['created_by'] = $this->site->getUser($inv->created_by);
		$this->data['page_title'] = lang("invoice");
		$this->session->set_userdata("invoice_status","");

		$this->load->view($this->theme . 'sales/payment_invoiceR', $this->data);
	}

	/* --------------------------------------------------------------------------------------------- */

    // =========================================================
// ENVIAR PARA AGT AUTOMATICAMENTE (FT/NC)
// =========================================================
private function _agt_auto_send($sale_id)
{
    $sale_id = (int)$sale_id;
    if (!$sale_id) return;

    // carrega o controller Agt para usar o mesmo código do endpoint
    $this->load->library('session'); // garante sessão OK
    $this->load->helper('url');

    // ⚠️ Chamada interna ao controller (não via HTTP)
    $this->load->library('router');

    // Carrega o controller Agt
    $this->load->library('../controllers/Agt');

    // Simula o ?send=1
    $_GET['send'] = 1;
    $_GET['debug'] = 0;

    // Executa
    $this->agt->registar_factura($sale_id);
}

public function get_pending_agt_ids()
{
    // se no teu sistema usas $this->loggedIn, mantém assim
    // se não usas, remove este bloco
    if (property_exists($this, 'loggedIn') && !$this->loggedIn) {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['ok'=>false,'error'=>'not_logged_in']);
        return;
    }

    // ✅ busca facturas pendentes: AGT != VALIDA/ENVIADO e não anuladas (mesa != NC)
    $this->db->select('id'); // <<<<<< se o campo for sale_id, mete: $this->db->select('sale_id AS id');
    $this->db->from('tec_sales');

    // ignora anuladas/NC
    $this->db->where('IFNULL(mesa,"") !=', 'NC');

    // só as que não estão validadas nem já enviadas
    $this->db->where("UPPER(IFNULL(agt_status,'')) NOT IN ('VALIDA','ENVIADO')", null, false);

    // opcional: se quiseres ignorar InvoiceNo a começar por NC
    // $this->db->where('UPPER(IFNULL(InvoiceNo,"")) NOT LIKE', 'NC%');

    // evita puxar milhares de uma vez
    $this->db->limit(500);

    $q = $this->db->get();
    $ids = [];
    foreach ($q->result() as $r) {
        $ids[] = (int)$r->id; // se select foi sale_id AS id, aqui fica igual
    }

    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['ok'=>true,'ids'=>$ids]);
}
public function get_pending_agt_sales_ids()
{
    header('Content-Type: application/json; charset=utf-8');

    // Busca IDs pendentes (não NC e AGT != validado)
    $this->db->select('id'); // se for sale_id: $this->db->select('sale_id AS id');
    $this->db->from('tec_sales');

    // ignora notas de crédito/anuladas
    $this->db->where('IFNULL(InvoiceType,"") !=', 'NC');

    // pendentes: tudo que não é VALIDA/VALIDADO
    $this->db->where("UPPER(IFNULL(agt_last_status,'')) NOT IN ('VALIDA','VALIDADO','V','VALIDATED')", null, false);

    $this->db->limit(500);

    $q = $this->db->get();
    $ids = [];

    foreach ($q->result() as $r) {
        $ids[] = (int)$r->id;
    }

    echo json_encode(['ok'=>true,'ids'=>$ids]);
}

}
